#!/bin/bash

# Queens Mountain VPS Deployment Script
# Run this on your HostGator VPS as root or with sudo

echo "🚀 Starting Queens Mountain Deployment..."

# Update system
echo "📦 Updating system packages..."
apt update && apt upgrade -y

# Install required packages
echo "📦 Installing required packages..."
apt install -y python3 python3-pip python3-venv nginx certbot python3-certbot-nginx mongodb ufw

# Install Node.js 18+ (required for React)
curl -fsSL https://deb.nodesource.com/setup_18.x | bash -
apt install -y nodejs

# Enable and start MongoDB
systemctl enable mongod
systemctl start mongod

# Create application directory
mkdir -p /var/www/queens-mountain
cd /var/www/queens-mountain

# Extract deployment files (upload deployment.zip first)
# unzip /path/to/deployment.zip

# Setup backend
echo "🔧 Setting up backend..."
cd backend
python3 -m venv venv
source venv/bin/activate
pip install -r requirements.txt

# Setup frontend
echo "🔧 Setting up frontend..."
cd ../frontend
npm install
npm run build

# Configure environment variables
echo "🔧 Creating environment configuration..."
cat > /var/www/queens-mountain/.env << EOF
# Database
MONGO_URL=mongodb://localhost:27017
DB_NAME=queens_mountain_prod

# JWT Secret (generate a secure random string)
JWT_SECRET=your-super-secure-jwt-secret-here

# Email Configuration (configure with your SMTP)
MAIL_USERNAME=your-email@domain.com
MAIL_PASSWORD=your-email-password
MAIL_FROM=noreply@yourdomain.com
MAIL_PORT=587
MAIL_SERVER=smtp.your-email-provider.com

# Stripe (add your keys)
STRIPE_API_KEY=sk_live_your_stripe_secret_key
STRIPE_PUBLISHABLE_KEY=pk_live_your_stripe_publishable_key

# Frontend URL
FRONTEND_URL=https://yourdomain.com

# Other settings
DEBUG=False
EOF

# Create systemd service for backend
echo "🔧 Creating systemd service..."
cat > /etc/systemd/system/queens-mountain.service << EOF
[Unit]
Description=Queens Mountain Backend
After=network.target

[Service]
User=www-data
Group=www-data
WorkingDirectory=/var/www/queens-mountain/backend
Environment=PATH=/var/www/queens-mountain/backend/venv/bin
ExecStart=/var/www/queens-mountain/backend/venv/bin/uvicorn server:app --host 0.0.0.0 --port 8000
Restart=always

[Install]
WantedBy=multi-user.target
EOF

# Configure Nginx
echo "🔧 Configuring Nginx..."
cat > /etc/nginx/sites-available/queens-mountain << EOF
server {
    listen 80;
    server_name yourdomain.com www.yourdomain.com;

    # Frontend
    location / {
        root /var/www/queens-mountain/frontend/build;
        index index.html index.htm;
        try_files \$uri \$uri/ /index.html;
    }

    # Backend API
    location /api {
        proxy_pass http://127.0.0.1:8000;
        proxy_set_header Host \$host;
        proxy_set_header X-Real-IP \$remote_addr;
        proxy_set_header X-Forwarded-For \$proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto \$scheme;
    }

    # Static files
    location /static {
        alias /var/www/queens-mountain/frontend/build/static;
        expires 1y;
        add_header Cache-Control "public, immutable";
    }
}
EOF

# Enable site
ln -s /etc/nginx/sites-available/queens-mountain /etc/nginx/sites-enabled/
rm /etc/nginx/sites-enabled/default

# Configure firewall
echo "🔒 Configuring firewall..."
ufw allow OpenSSH
ufw allow 'Nginx Full'
ufw --force enable

# Start services
echo "🚀 Starting services..."
systemctl daemon-reload
systemctl enable queens-mountain
systemctl start queens-mountain
systemctl restart nginx

# SSL Certificate (optional but recommended)
echo "🔒 Setting up SSL (optional)..."
# certbot --nginx -d yourdomain.com -d www.yourdomain.com

echo "✅ Deployment complete!"
echo "🌐 Your site should be available at: http://your-vps-ip"
echo "🔧 Don't forget to:"
echo "   1. Update the .env file with your actual credentials"
echo "   2. Configure your domain DNS to point to this VPS"
echo "   3. Set up SSL certificate with certbot if needed"
echo "   4. Test the application thoroughly"